/*******************************************************************************
 * Copyright (c) 2004 Duke University
 *
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.cs.duke.edu/csed/ambient/copyright.html
 * 
*******************************************************************************/
/*******************************************************************************
 * ResourceTreeAndListGroup
 * 
 *  org.eclipse.ui.internal.dialogs;
 *******************************************************************************/
package edu.duke.ambient.ui.checkin;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import org.eclipse.jface.util.ListenerList;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Tree;

import edu.duke.ambient.ui.checkin.pages.AmbientCheckoutPage;


/**
 *	Workbench-level composite that combines a CheckboxTreeViewer and CheckboxListViewer.
 *	All myViewer selection-driven interactions are handled within this object
 */
public class AmbientResourceTreeGroup implements ISelectionProvider {
	private Object root;
	private Object currentTreeSelection;
	private Collection expandedTreeNodes = new HashSet();
	private Map checkedStateStore = new HashMap(9);
	private Collection whiteCheckedTreeItems = new HashSet();
	private ListenerList listeners = new ListenerList();

	private ITreeContentProvider treeContentProvider;
	private IStructuredContentProvider listContentProvider;
	private ILabelProvider treeLabelProvider;
 	private ILabelProvider listLabelProvider;
	private CheckboxTreeViewer treeViewer;
	private Shared mySharedFunctions = new Shared();
	private static int PREFERRED_HEIGHT = 150;
	private AmbientCheckoutPage myCaller;
	
	private final boolean DEBUG = false;
	
	/**
	 *	Create an instance of this class.  Use this constructor if you wish to specify
	 *	the width and/or height of the combined widget (to only hardcode one of the
	 *	sizing dimensions, specify the other dimension's value as -1)
	 */
	
	public AmbientResourceTreeGroup(Composite parent,
		Object rootObject,ITreeContentProvider treeContentProvider,
		ILabelProvider treeLabelProvider,
		IStructuredContentProvider listContentProvider,
		ILabelProvider listLabelProvider,int style,boolean useHeightHint) {

		root = rootObject;
		this.treeContentProvider = treeContentProvider;
		this.listContentProvider = listContentProvider;
		this.treeLabelProvider = new AmbientTreeLabelProvider();//treeLabelProvider;
		this.listLabelProvider = listLabelProvider;
		
		if (DEBUG){
			System.out.println("In AmbientResourceTreeGroup: ");
			System.out.println("     about to create contents");
		}
		createContents(parent, style, useHeightHint);
	}
	/**
	 * This method must be called just before this window becomes visible.
	 */
	public void aboutToOpen() {
		currentTreeSelection = null;

		//select the first element in the list
		Object[] elements = treeContentProvider.getElements(root);
		Object primary = elements.length > 0 ? elements[0] : null;
		if (primary != null) {
			treeViewer.setSelection(new StructuredSelection(primary));
		}
		treeViewer.getControl().setFocus();
	}

	/**
	 *	Lay out and initialize self's visual components.
	 *
	 *	@param parent org.eclipse.swt.widgets.Composite
	 *	@param useHeightHint. If true yse the preferredHeight.
	 */
	protected void createContents(Composite parent,int style,
			boolean useHeightHint) {
		// group pane
		Composite composite = new Composite(parent, style);
		composite.setFont(parent.getFont());
		GridLayout layout = new GridLayout();
		layout.numColumns = 1;
		layout.makeColumnsEqualWidth = true;
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		composite.setLayout(layout);
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));

		createTreeViewer(composite, useHeightHint);
		initialize();
	}

	/**
	 *	Create this group's tree myViewer.
	 */
	protected void createTreeViewer(Composite parent, boolean useHeightHint) {
		final Tree tree = new Tree(parent,SWT.BORDER);
		GridData data = new GridData(GridData.FILL_BOTH);
		if (useHeightHint)
			data.heightHint = PREFERRED_HEIGHT;
		tree.setLayoutData(data);
		tree.setFont(parent.getFont());

		treeViewer = new CheckboxTreeViewer(tree);
		treeViewer.setContentProvider(treeContentProvider);
		treeViewer.setLabelProvider(treeLabelProvider);
		/*
		tree.addSelectionListener(new SelectionListener(){

			public void widgetSelected(SelectionEvent e) {
				try{
					Object[] selected =
						((IStructuredSelection) (treeViewer.getSelection())).toArray();
					ICVSRemoteFolder selection = (ICVSRemoteFolder) selected[0];
					
					if (DEBUG){
						System.out.println("selected " + 
								selection.getName());
					}
				}catch(Exception ex){}
				
			}

			public void widgetDefaultSelected(SelectionEvent e) {
			}
		});
		*/
	}

	public List getSelectedResources() {
		List resourcesToExport = new ArrayList();
		
		Object[] resourceToAdd =
			((IStructuredSelection) (treeViewer.getSelection())).toArray();
		
		if (resourceToAdd.length > 0)
			resourcesToExport.add(resourceToAdd[0]);

		return resourcesToExport;
	}

	/**
	 * Get the full label of the treeElement (its name and its parent's name).
	 * @param treeElement - the element being exported
	 * @param parentLabel - the label of the parent, can be null
	 * @return String
	 */
	protected String getFullLabel(Object treeElement, String parentLabel) {
		String parentName = parentLabel;

		if (parentLabel == null)
			parentName = ""; //$NON-NLS-1$

		if (parentName.length() > 0
			&& (!parentName.endsWith(File.separator))) {
			parentName += File.separatorChar;
		}
		return parentName + treeLabelProvider.getText(treeElement);
	}
	/**
	 *	Return a count of the number of list items associated with a
	 *	given tree item.
	 *
	 *	@return int
	 *	@param treeElement java.lang.Object
	 */
	protected int getListItemsSize(Object treeElement) {
		Object[] elements = listContentProvider.getElements(treeElement);
		return elements.length;
	}

	/**
	 *	Initialize this group's viewers after they have been laid out.
	 */
	protected void initialize() {
		treeViewer.setInput(root);
		this.expandedTreeNodes = new ArrayList();
		this.expandedTreeNodes.add(root);
	}

	/**
	 * Set the root of the widget to be new Root. Regenerate all of the tables and lists from this
	 * value.
	 * @param newRoot 
	 */
	public void setRoot(Object newRoot) {
		this.root = newRoot;
		initialize();
	}

	/**
	 *	Set the tree myViewer's providers to those passed
	 *
	 *	@param contentProvider ITreeContentProvider
	 *	@param labelProvider ILabelProvider
	 */
	public void setTreeProviders(ITreeContentProvider contentProvider,
		       ILabelProvider labelProvider) {
		treeViewer.setContentProvider(contentProvider);
		treeViewer.setLabelProvider(labelProvider);
	}
	public void setTreeSorter(ViewerSorter sorter) {
		treeViewer.setSorter(sorter);
	}

	public void setFocus() {
		this.treeViewer.getTree().setFocus();
	}
	
	public void addSelectionChangedListener(ISelectionChangedListener listener) {
		treeViewer.addSelectionChangedListener(listener);
	}
	
	public ISelection getSelection() {
		return treeViewer.getSelection();
	}
	
	public void removeSelectionChangedListener(ISelectionChangedListener listener) {
		treeViewer.removeSelectionChangedListener(listener);
		
	}
	public void setSelection(ISelection selection) {
		//not needed
	}
}
