/*******************************************************************************
 * Copyright (c) 2004 Duke University
 *
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.cs.duke.edu/csed/ambient/copyright.html
 * 
*******************************************************************************/
/*******************************************************************************
 * This class is edited from the original WizardFileSystemResourceExportPage1 in
 * org.eclipse.ui.wizards.datatransfer
 *******************************************************************************/
package edu.duke.ambient.ui.checkin;

import java.lang.reflect.InvocationTargetException;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.team.core.TeamException;
import org.eclipse.team.internal.ccvs.core.CVSException;
import org.eclipse.team.internal.ccvs.core.ICVSFolder;
import org.eclipse.team.internal.ccvs.core.ICVSRemoteFile;
import org.eclipse.team.internal.ccvs.core.ICVSRemoteResource;
import org.eclipse.team.internal.ccvs.core.ICVSResource;
import org.eclipse.team.internal.ccvs.core.ILogEntry;
import org.eclipse.team.internal.ccvs.core.resources.CVSWorkspaceRoot;
import org.eclipse.team.internal.ccvs.ui.CVSUIPlugin;
import org.eclipse.team.internal.ccvs.ui.HistoryTableProvider;
import org.eclipse.team.internal.ccvs.ui.repo.RepositoryManager;
import org.eclipse.ui.actions.WorkspaceModifyOperation;

import edu.duke.ambient.ui.checkin.pages.AmbientExportResourcesPage;

/**
 * @author Frank Mueller
 *
 */
public class AmbientVerificationPage extends AmbientExportResourcesPage {

	// Constants for determining the type of progress. Subclasses may
	// pass one of these values to the run method.
	public final static int PROGRESS_DIALOG = 1;
	public final static int PROGRESS_BUSYCURSOR = 2;

	private HistoryTableProvider historyTableProvider;

	private String result;

	public void setResult(final String r) {
		result += new String(r);
	}

	public AmbientVerificationPage() {
		this("AmbientConfirmPage", null);
		setTitle("Verify Project Submission");
		setDescription("Please select a project to verify:");
		historyTableProvider = new HistoryTableProvider();
		result =
			"These files have been last updated on the server at the following times\n\n"
				+ "Date\t\tTime\t\tFile Name\n";
	}

	/**
	 * Constructor
	 * @param pageName
	 * @param selection
	 */
	protected AmbientVerificationPage(
		String pageName,
		IStructuredSelection selection) {
		super(pageName, selection);
	}

	/**
	 * @return
	 */
	public boolean finish() {

		List resourcesTryingToConfirm = getWhiteCheckedResources();
		//	Save dirty editors if possible but do not stop if not all are saved
		saveDirtyEditors();

		if (resourcesTryingToConfirm.size() <= 0) {
			MessageDialog.openInformation(
				getContainer().getShell(),
				"Error",
				"No Project Name Selected");
			return false;
		}

		//simply convert the List back to an array
		IResource[] resourcesToConfirm =
			new IResource[resourcesTryingToConfirm.size()];
		for (int i = 0; i < resourcesToConfirm.length; i++) {
			resourcesToConfirm[i] = (IResource) resourcesTryingToConfirm.get(i);
		}

		//find only those resources that are shared with CVS repo
		try {
			resourcesToConfirm = getSharedResources(resourcesToConfirm);
		} catch (CVSException e) {
			return false;
		}

		if (resourcesToConfirm.length <= 0) {
			MessageDialog.openError(
				getContainer().getShell(),
				"Error",
				"The project has never been submitted");
			return false;
		}
 
		try {
			verifyProject(resourcesToConfirm);
		} catch (InvocationTargetException e1) {
			return false;
		} catch (InterruptedException e1) {
			return false;
		}

		MessageDialog.openInformation(
			getContainer().getShell(),
			"Submission Confirmation",
			result);
		return true;
	}

	protected void verifyProject(final IResource[] resourcesToConfirm)
		throws InvocationTargetException, InterruptedException {

		run(new WorkspaceModifyOperation() {
			public void execute(IProgressMonitor monitor) {
				monitor.beginTask(null, 100);
				//System.out.println(monitor.getClass());
				String result = "";
				for (int i = 0; i < resourcesToConfirm.length; i++) {

					ICVSRemoteResource cvsResource;
					try {
						cvsResource =
							CVSWorkspaceRoot.getRemoteResourceFor(
								resourcesToConfirm[i]);
					} catch (CVSException e1) {
						return; // false;
					}
					ICVSRemoteFile remoteFile = (ICVSRemoteFile) cvsResource;
					monitor.setTaskName(remoteFile.getName());

					ILogEntry[] log;
					try {
						log =
							remoteFile.getLogEntries(new NullProgressMonitor());
					} catch (TeamException e2) {
						return; // false;
					}
					Date time = null;
					String name = remoteFile.getName();
					String remoteVersion = "";
					String localVersion = "";
					try {
						localVersion = remoteFile.getRevision();
					} catch (TeamException e4) {
						return; // false;
					}
					if (log[0] != null) {
						time = log[0].getDate();
						remoteVersion = log[0].getRevision();
					}

					result += DateFormat.getDateInstance().format(time)
						+ " \t"
						+ DateFormat.getTimeInstance().format(time)
						+ "\t"
						+ name
						+ "\n";

					monitor.worked((100 * (i + 1)) / resourcesToConfirm.length);
				}
				setResult(result);
				monitor.done();
			}
		}, true /* cancelable */
		, PROGRESS_DIALOG);

	}

	protected RepositoryManager getRepositoryManager() {
		return CVSUIPlugin.getPlugin().getRepositoryManager();
	}

	/*
	 * Return all resources in the provided collection that are shared with a repo
	 * @param resources
	 * @return IResource[]
	 */
	private IResource[] getSharedResources(IResource[] resources)
		throws CVSException {
		List shared = new ArrayList();
		for (int i = 0; i < resources.length; i++) {
			IResource resource = resources[i];
			ICVSResource cvsResource =
				CVSWorkspaceRoot.getCVSResourceFor(resource);
			if (cvsResource.isManaged()
				|| (cvsResource.isFolder()
					&& ((ICVSFolder) cvsResource).isCVSFolder())) {
				shared.add(resource);
			}
		}
		return (IResource[]) shared.toArray(new IResource[shared.size()]);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.dialogs.WizardDataTransferPage#handleEvent(org.eclipse.swt.widgets.Event)
	 */
	public void handleEvent(Event arg0) {
		// Auto-generated method stub

	}

	/* (non-Javadoc)
	 * @see edu.purdue.Ambient.ui.submit.AmbientExportResourcesPage#createDestinationGroup(org.eclipse.swt.widgets.Composite)
	 */
	protected void createDestinationGroup(Composite parent) {
		// Auto-generated method stub

	}

	final protected void run(
		final IRunnableWithProgress runnable,
		boolean cancelable,
		int progressKind)
		throws InvocationTargetException, InterruptedException {
		final Exception[] exceptions = new Exception[] { null };

		// Ensure that no repository view refresh happens until after the action
		final IRunnableWithProgress innerRunnable =
			new IRunnableWithProgress() {
			public void run(IProgressMonitor monitor)
				throws InvocationTargetException, InterruptedException {
				getRepositoryManager().run(runnable, monitor);
			}
		};

		switch (progressKind) {
			case PROGRESS_DIALOG :
			default :
				new ProgressMonitorDialog(getShell()).run(
					cancelable,
					true,
					innerRunnable);
				break;
		}
		if (exceptions[0] != null) {
			if (exceptions[0] instanceof InvocationTargetException)
				throw (InvocationTargetException) exceptions[0];
			else
				throw (InterruptedException) exceptions[0];
		}
	}

}
