/*******************************************************************************
 * Copyright (c) 2004 Duke University
 *
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.cs.duke.edu/csed/ambient/copyright.html
 * 
*******************************************************************************/
/*******************************************************************************
 * TypeHierarchyTransferDropAdapter
 * 
 * Contributors:
 *     org.eclipse.jdt.internal.ui.typehierarchy;
 *******************************************************************************/
package edu.duke.ambient.ui.hierarchy;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.jdt.core.IBuffer;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.ISourceRange;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.ToolFactory;
import org.eclipse.jdt.core.compiler.IScanner;
import org.eclipse.jdt.core.compiler.ITerminalSymbols;
import org.eclipse.jdt.core.compiler.InvalidInputException;
import org.eclipse.jdt.internal.core.CompilationUnit;
import org.eclipse.jdt.internal.core.SourceType;
import org.eclipse.jdt.internal.ui.javaeditor.JavaEditor;
import org.eclipse.jdt.internal.ui.packageview.SelectionTransferDropAdapter;
import org.eclipse.jdt.internal.ui.util.OpenTypeHierarchyUtil;
import org.eclipse.jdt.internal.ui.util.SelectionUtil;
import org.eclipse.jdt.ui.actions.OrganizeImportsAction;
import org.eclipse.jface.viewers.AbstractTreeViewer;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IReusableEditor;
import org.eclipse.ui.IWorkbenchPage;

import edu.duke.ambient.AmbientPlugin;

public class AmbientHierarchyDragAndDropAdapter
	extends SelectionTransferDropAdapter {

	private static final int OPERATION = DND.DROP_LINK;

	private static IJavaElement getInputElement(ISelection selection) {
		Object single = SelectionUtil.getSingleElement(selection);
		if (single == null)
			return null;
		IJavaElement[] candidates = OpenTypeHierarchyUtil.getCandidates(single);
		if (candidates != null && candidates.length > 0)
			return candidates[0];
		return null;
	}
	private AmbientHierarchyView fTypeHierarchyViewPart;

	public AmbientHierarchyDragAndDropAdapter(
		AmbientHierarchyView viewPart,
		AbstractTreeViewer viewer) {
		super(viewer);
		fTypeHierarchyViewPart = viewPart;
	}

	public void drop(Object target, DropTargetEvent event) {
		//save all files before making and modifications
		IWorkbenchPage page =
			AmbientPlugin.getActiveWorkbenchWindow().getActivePage();
		if (page != null)
			page.saveAllEditors(false);

		//need to make a distinction between the different cases
		//case 1: source = class; target = class => source extends target
		//case 2: source = class; target = null=> source extends nothing (actually this depends on the branch source was dragged from)
		//case 3: source = class; target = interface => source implements target
		//case 4: source = interface; target = null => interface source extends nothing
		//case 5: source = interface; target = interface => interface extends interface
		//case 6: source = interface; target = class => don't do anything

		Object source =
			((StructuredSelection) getSelection()).getFirstElement();

		try {
			if (source instanceof SourceType) {
				SourceType sourceType = ((SourceType) source);

				if (target == null) {
					//case 4
					if (sourceType.isInterface()) {
						resetInterface(sourceType);
						//case 2
					} else if (sourceType.isClass()) {
						//we need to figure out if we need to remove a class extenstion 
						//or interface inheritance

						resetClass(sourceType);
					}
				} else {
					SourceType targetType = ((SourceType) target);
					//case 5
					if (sourceType.isInterface() && targetType.isInterface()) {
						extendInterface(target);
						//case 1
					} else if (sourceType.isClass() && targetType.isClass()) {
						extendClass(target);
						//case 3
					} else if (
						sourceType.isClass() && targetType.isInterface()) {
						//check if source already implements the class
						String[] currentInterfaces =
							sourceType.getSuperInterfaceNames();
						for (int i = 0; i < currentInterfaces.length; i++) {
							if (currentInterfaces[i]
								.equals(targetType.getElementName()))
								return;
						}
						implementInterface(target, currentInterfaces.length);
					}
				}
			}
		} catch (JavaModelException e) {
			e.printStackTrace();
		}

		//refresh all editors
		IEditorPart editor = page.getActiveEditor();
		page.reuseEditor((IReusableEditor) editor, editor.getEditorInput());
		
		//organize imports
		
		OrganizeImportsAction oia = new OrganizeImportsAction((JavaEditor)editor);	
		oia.run();
		editor.doSave(new NullProgressMonitor());
	}

	public void extendInterface(Object target) {

		Object source =
			((StructuredSelection) getSelection()).getFirstElement();

		if (source instanceof SourceType && target instanceof SourceType) {

			CompilationUnit javaFile =
				(CompilationUnit) (((SourceType) source).getParent());

			try {
				specifiyClass(
					((SourceType) source).readableName(),
					ITerminalSymbols.TokenNameinterface,
					((SourceType) target).readableName(),
					new NullProgressMonitor(),
					javaFile);
			} catch (JavaModelException e) {
				e.printStackTrace();
			} catch (InvalidInputException e) {
				e.printStackTrace();
			}
		}
	}

	public void extendClass(Object target) {

		Object source =
			((StructuredSelection) getSelection()).getFirstElement();

		if (source instanceof SourceType && target instanceof SourceType) {
			//System.out.println(
			//	((SourceType) source).readableName()
			//		+ " extends "
			//		+ ((SourceType) target).readableName());

			CompilationUnit javaFile =
				(CompilationUnit) (((SourceType) source).getParent());

			try {
				specifiyClass(
					((SourceType) source).readableName(),
					ITerminalSymbols.TokenNameclass,
					((SourceType) target).readableName(),
					new NullProgressMonitor(),
					javaFile);
			} catch (JavaModelException e) {
				e.printStackTrace();
			} catch (InvalidInputException e) {
				e.printStackTrace();
			}
		}
	}

	public void implementInterface(Object target, int numOfInt) {

		Object source =
			((StructuredSelection) getSelection()).getFirstElement();

		if (source instanceof SourceType && target instanceof SourceType) {
			try {
				CompilationUnit javaFile =
					(CompilationUnit) (((SourceType) source).getParent());
				String superClass = ((SourceType) source).getSuperclassName();

				specifiyInterface(
					((SourceType) source).readableName(),
					numOfInt,
					superClass,
					((SourceType) target).readableName(),
					new NullProgressMonitor(),
					javaFile);
			} catch (JavaModelException e) {
				e.printStackTrace();
			} catch (InvalidInputException e) {
				e.printStackTrace();
			}
		}
	}

	public void resetClass(SourceType source) {

		CompilationUnit javaFile = (CompilationUnit) ((source).getParent());

		try {
			specifiyClass(
				((SourceType) source).readableName(),
				ITerminalSymbols.TokenNameclass,
				null,
				new NullProgressMonitor(),
				javaFile);
		} catch (JavaModelException e) {
			e.printStackTrace();
		} catch (InvalidInputException e) {
			e.printStackTrace();
		}

	}

	public void resetInterface(SourceType source) {

		CompilationUnit javaFile = (CompilationUnit) ((source).getParent());

		try {
			String superClass = source.getSuperclassName();
			specifiyClass(
				((SourceType) source).readableName(),
				ITerminalSymbols.TokenNameinterface,
				null,
				new NullProgressMonitor(),
				javaFile);
		} catch (JavaModelException e) {
			e.printStackTrace();
		} catch (InvalidInputException e) {
			e.printStackTrace();
		}

	}

	/*
	 * sets up the scanner we need to analyze the source code file
	 */
	public IScanner prepareScanner(CompilationUnit cu, IBuffer buffer)
		throws JavaModelException, InvalidInputException {
		IScanner scanner =
			ToolFactory.createScanner(false, false, false, false);
		scanner.setSource(buffer.getCharacters());
		ISourceRange sr = cu.getSourceRange();
		scanner.resetTo(sr.getOffset(), sr.getOffset() + sr.getLength() - 1);
		return scanner;
	}

	public void specifiyClass(
		String source,
		int sourceType,
		String target,
		IProgressMonitor monitor,
		ICompilationUnit parentElement)
		throws JavaModelException, InvalidInputException {

		ICompilationUnit cu = ((ICompilationUnit) parentElement).getWorkingCopy(null);
		if (cu.isWorkingCopy()) {
			IBuffer buffer = cu.getBuffer();
			IScanner scanner = prepareScanner((CompilationUnit)cu, buffer);

			int start = -1;
			int end = -1;
			int token = scanner.getNextToken();
			while (token != ITerminalSymbols.TokenNameEOF) {
				token = scanner.getNextToken();

				//match keyword class
				if (token == sourceType) {
					end = scanner.getCurrentTokenEndPosition();
				} else {
					//match the source class
					if (new String(scanner.getCurrentTokenSource())
						.equals(source)
						&& (end + 2) == scanner.getCurrentTokenStartPosition()) {
						end = scanner.getCurrentTokenEndPosition();

					} else {
						//match keyword extends
						if (new String(scanner.getCurrentTokenSource())
							.equals("extends")
							&& (end + 2)
								== scanner.getCurrentTokenStartPosition()) {
							start = scanner.getCurrentTokenStartPosition() - 1;
							token = scanner.getNextToken();

							if (target != null) {
								buffer.replace(
									start,
									scanner.getCurrentTokenEndPosition()
										- start
										+ 1,
									" extends " + target);
								break;
							} else {
								while (new String(scanner
									.getCurrentTokenSource())
									.equals("{")
									== false) {
									scanner.getNextToken();
								}
								buffer.replace(
									start,
									scanner.getCurrentTokenEndPosition()
										- start,
									" ");
								break;
							}

						} else //match keyword implements
							if (new String(scanner.getCurrentTokenSource())
								.equals("implements")
								&& (end + 2)
									== scanner.getCurrentTokenStartPosition()
								&& target == null) {
									
								start = scanner.getCurrentTokenStartPosition() - 1;
								while (new String(scanner
									.getCurrentTokenSource())
									.equals("{")
									== false) {
									scanner.getNextToken();
								}
								buffer.replace(
									start,
									scanner.getCurrentTokenEndPosition()
										- start,
									" ");
								break;
							} else {
								if (target != null && end != -1) {
									//this used to be an Object class
									buffer.replace(
										end + 1,
										0,
										" extends " + target);
									break;
								} else {
									start = -1;
									end = -1;
								}
							}

					}
				}
			}
		}

		((CompilationUnit) cu).commitWorkingCopy(true, monitor);

	}

	public void specifiyInterface(
		String source,
		int numOfInt,
		String sourceExtends,
		String target,
		IProgressMonitor monitor,
		ICompilationUnit parentElement)
		throws JavaModelException, InvalidInputException {

		ICompilationUnit cu = ((ICompilationUnit) parentElement).getWorkingCopy(null);
		if (cu.isWorkingCopy()) {

			IBuffer buffer = cu.getBuffer();
			IScanner scanner = prepareScanner((CompilationUnit)cu, buffer);

			int start = -1;
			int end = -1;
			int token = scanner.getNextToken();
			while (token != ITerminalSymbols.TokenNameEOF) {
				token = scanner.getNextToken();

				//match keyword class
				if (token == ITerminalSymbols.TokenNameclass) {
					end = scanner.getCurrentTokenEndPosition();
				} else {
					//match the source class
					if (new String(scanner.getCurrentTokenSource())
						.equals(source)
						&& (end + 2) == scanner.getCurrentTokenStartPosition()) {
						end = scanner.getCurrentTokenEndPosition();

					} else {
						if (sourceExtends != null) {
							//match keyword extends
							if (new String(scanner.getCurrentTokenSource())
								.equals("extends")
								&& (end + 2)
									== scanner.getCurrentTokenStartPosition()) {
								end = scanner.getCurrentTokenEndPosition();
							} else {
								//match super class
								if (new String(scanner.getCurrentTokenSource())
									.equals(sourceExtends)
									&& (end + 2)
										== scanner
											.getCurrentTokenStartPosition()) {
									end = scanner.getCurrentTokenEndPosition();
								} else {
									//match keyword implements
									if (new String(scanner
										.getCurrentTokenSource())
										.equals("implements")
										&& (end + 2)
											== scanner
												.getCurrentTokenStartPosition()) {

										start =
											scanner
												.getCurrentTokenStartPosition()
												- 1;
										token = scanner.getNextToken();

										if (target != null) {
											if (numOfInt > 0)
												buffer.replace(
													start,
													scanner
														.getCurrentTokenStartPosition()
														- start,
													" implements "
														+ target
														+ ", ");
											else
												buffer.replace(
													start,
													scanner
														.getCurrentTokenEndPosition()
														- start
														+ 1,
													" implements " + target);
											break;
										} else {
											buffer.replace(
												start,
												scanner
													.getCurrentTokenEndPosition()
													- start
													+ 1,
												"");
											break;
										}

									} else {
										if (target != null && end != -1) {
											//this used to be an Object class
											buffer.replace(
												end + 1,
												0,
												" implements " + target);
											break;
										} else {
											start = -1;
											end = -1;
										}
									}
								}
							}
						} else {
							//match keyword implements
							if (new String(scanner.getCurrentTokenSource())
								.equals("implements")
								&& (end + 2)
									== scanner.getCurrentTokenStartPosition()) {
								start =
									scanner.getCurrentTokenStartPosition() - 1;
								token = scanner.getNextToken();

								if (target != null) {
									if (numOfInt > 0)
										buffer.replace(
											start,
											scanner
												.getCurrentTokenStartPosition()
												- start,
											" implements " + target + ", ");
									else
										buffer.replace(
											start,
											scanner
												.getCurrentTokenEndPosition()
												- start
												+ 1,
											" implements " + target);
									break;
								} else {
									buffer.replace(
										start,
										scanner.getCurrentTokenEndPosition()
											- start
											+ 1,
										"");
									break;
								}

							} else {
								if (target != null && end != -1) {
									//this used to be an Object class
									buffer.replace(
										end + 1,
										0,
										" implements " + target);
									break;
								} else {
									start = -1;
									end = -1;
								}
							}
						}
					}
				}
			}
		}

		((CompilationUnit) cu).commitWorkingCopy(true, monitor);

	}

	public void validateDrop(
		Object target,
		DropTargetEvent event,
		int operation) {

		event.detail = DND.DROP_NONE;
		initializeSelection();
		if (target != null) {
			super.validateDrop(target, event, operation);
			return;
		}
		if (getInputElement(getSelection()) != null)
			event.detail = AmbientHierarchyDragAndDropAdapter.OPERATION;
	}

}
