/*******************************************************************************
 * Copyright (c) 2004 Duke University
 *
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.cs.duke.edu/csed/ambient/copyright.html
 * 
*******************************************************************************/
/*******************************************************************************
 * This class is edited from the original TypeHierarchyViewPart in
 * org.eclipse.jdt.internal.ui.typehierarchy
 *******************************************************************************/
package edu.duke.ambient.ui.hierarchy;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IMember;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.internal.corext.util.JavaModelUtil;
import org.eclipse.jdt.internal.ui.IJavaHelpContextIds;
import org.eclipse.jdt.internal.ui.JavaPlugin;
import org.eclipse.jdt.internal.ui.actions.AddMethodStubAction;
import org.eclipse.jdt.internal.ui.actions.CompositeActionGroup;
import org.eclipse.jdt.internal.ui.actions.NewWizardsActionGroup;
import org.eclipse.jdt.internal.ui.dnd.DelegatingDropAdapter;
import org.eclipse.jdt.internal.ui.dnd.JdtViewerDragAdapter;
import org.eclipse.ui.views.navigator.LocalSelectionTransfer;
import org.eclipse.jdt.internal.ui.dnd.TransferDragSourceListener;
import org.eclipse.jdt.internal.ui.dnd.TransferDropTargetListener;
import org.eclipse.jdt.internal.ui.javaeditor.EditorUtility;
import org.eclipse.jdt.internal.ui.packageview.SelectionTransferDragAdapter;
import org.eclipse.jdt.internal.ui.typehierarchy.ITypeHierarchyLifeCycleListener;
import org.eclipse.jdt.internal.ui.typehierarchy.SelectionProviderMediator;
import org.eclipse.jdt.internal.ui.typehierarchy.TypeHierarchyLifeCycle;
import org.eclipse.jdt.internal.ui.typehierarchy.TypeHierarchyMessages;
import org.eclipse.jdt.internal.ui.util.ExceptionHandler;
import org.eclipse.jdt.internal.ui.viewsupport.IViewPartInputProvider;
import org.eclipse.jdt.internal.ui.viewsupport.JavaElementLabels;
import org.eclipse.jdt.internal.ui.viewsupport.JavaUILabelProvider;
import org.eclipse.jdt.internal.ui.viewsupport.StatusBarUpdater;
import org.eclipse.jdt.ui.IContextMenuConstants;
import org.eclipse.jdt.ui.ITypeHierarchyViewPart;
import org.eclipse.jdt.ui.JavaUI;
//import org.eclipse.jdt.ui.PreferenceConstants;
import org.eclipse.jdt.ui.actions.CCPActionGroup;
import org.eclipse.jdt.ui.actions.GenerateActionGroup;
import org.eclipse.jdt.ui.actions.JavaSearchActionGroup;
import org.eclipse.jdt.ui.actions.OpenEditorActionGroup;
import org.eclipse.jdt.ui.actions.OpenViewActionGroup;
import org.eclipse.jdt.ui.actions.RefactorActionGroup;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.util.Assert;
import org.eclipse.jface.viewers.AbstractTreeViewer;
import org.eclipse.jface.viewers.IBasicPropertyConstants;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.custom.ViewForm;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DropTarget;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IPageLayout;
import org.eclipse.ui.IPartListener2;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartReference;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.actions.ActionContext;
import org.eclipse.ui.actions.ActionGroup;
import org.eclipse.ui.help.WorkbenchHelp;
import org.eclipse.ui.part.IShowInSource;
import org.eclipse.ui.part.IShowInTargetList;
import org.eclipse.ui.part.PageBook;
import org.eclipse.ui.part.ShowInContext;
import org.eclipse.ui.part.ViewPart;

/**
 * view showing the supertypes/subtypes of its input.
 */
public class AmbientHierarchyView
	extends ViewPart
	implements ITypeHierarchyViewPart, IViewPartInputProvider {

	public static final int VIEW_ID_TYPE = 0;

	private static final String DIALOGSTORE_HIERARCHYVIEW = "TypeHierarchyViewPart.hierarchyview"; //$NON-NLS-1$

	private static final String TAG_INPUT = "input"; //$NON-NLS-1$
	private static final String TAG_VIEW = "view"; //$NON-NLS-1$
	private static final String TAG_ORIENTATION = "orientation"; //$NON-NLS-1$
	private static final String TAG_RATIO = "ratio"; //$NON-NLS-1$
	private static final String TAG_SELECTION = "selection"; //$NON-NLS-1$
	private static final String TAG_VERTICAL_SCROLL = "vertical_scroll"; //$NON-NLS-1$

	private static final String GROUP_FOCUS = "group.focus"; //$NON-NLS-1$

	// the selected type in the hierarchy view
	private IType fSelectedType;
	// input element or null
	private IJavaElement fInputElement;
	private IDialogSettings fDialogSettings;

	private TypeHierarchyLifeCycle fHierarchyLifeCycle;
	private ITypeHierarchyLifeCycleListener fTypeHierarchyLifeCycleListener;

	private SelectionProviderMediator fSelectionProviderMediator;
	private ISelectionChangedListener fSelectionChangedListener;
	private IPartListener2 fPartListener;

	private boolean fIsVisible;
	private boolean fNeedRefresh;
	private boolean fIsRefreshRunnablePosted;

	private int fCurrentViewerIndex;
	private AmbientHierarchyViewer[] fAllViewers;

	private SashForm fTypeMethodsSplitter;
	private PageBook fViewerbook;
	private PageBook fPagebook;

	private Label fNoHierarchyShownLabel;
	private Label fEmptyTypesViewer;

	private ViewForm fTypeViewerViewForm;
	private JavaUILabelProvider fPaneLabelProvider;

	private AddMethodStubAction fAddStubAction;
	private CompositeActionGroup fActionGroups;
	private CCPActionGroup fCCPActionGroup;

	public AmbientHierarchyView() {
		fSelectedType = null;
		fInputElement = null;
		fIsVisible = false;
		fIsRefreshRunnablePosted = false;

		/*boolean isReconciled =
			PreferenceConstants.UPDATE_WHILE_EDITING.equals(
				PreferenceConstants.getPreferenceStore().getString(
					PreferenceConstants.UPDATE_JAVA_VIEWS));
		*/
		fHierarchyLifeCycle = new TypeHierarchyLifeCycle();
		//fHierarchyLifeCycle.setReconciled(isReconciled);
		fTypeHierarchyLifeCycleListener =
			new ITypeHierarchyLifeCycleListener() {
			public void typeHierarchyChanged(
				TypeHierarchyLifeCycle typeHierarchy,
				IType[] changedTypes) {
				doTypeHierarchyChanged(typeHierarchy, changedTypes);
			}
		};
		fHierarchyLifeCycle.addChangedListener(fTypeHierarchyLifeCycleListener);

		fAllViewers = null;

		fDialogSettings = JavaPlugin.getDefault().getDialogSettings();

		fPaneLabelProvider = new JavaUILabelProvider();

		fAddStubAction = new AddMethodStubAction();

		fPartListener = new IPartListener2() {
			public void partVisible(IWorkbenchPartReference ref) {
				IWorkbenchPart part = ref.getPart(false);
				if (part == AmbientHierarchyView.this) {
					visibilityChanged(true);
				}
			}

			public void partHidden(IWorkbenchPartReference ref) {
				IWorkbenchPart part = ref.getPart(false);
				if (part == AmbientHierarchyView.this) {
					visibilityChanged(false);
				}
			}

			public void partActivated(IWorkbenchPartReference ref) {
				IWorkbenchPart part = ref.getPart(false);
			}

			public void partInputChanged(IWorkbenchPartReference ref) {
				IWorkbenchPart part = ref.getPart(false);
			};

			public void partBroughtToTop(IWorkbenchPartReference ref) {
			}
			public void partClosed(IWorkbenchPartReference ref) {
			}
			public void partDeactivated(IWorkbenchPartReference ref) {
			}
			public void partOpened(IWorkbenchPartReference ref) {
			}
		};

		fSelectionChangedListener = new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				doSelectionChanged(event);
			}
		};
	}

	/**
	 * Selects an member in the methods list or in the current hierarchy.
	 */
	public void selectMember(IMember member) {
		if (member.getElementType() != IJavaElement.TYPE) {
			// methods are working copies
		/*	if (fHierarchyLifeCycle.isReconciled()) {
				member = JavaModelUtil.toWorkingCopy(member);
			}*/
		} else {
			Control viewerControl = getCurrentViewer().getControl();
			if (viewerControl != null && !viewerControl.isDisposed()) {
				viewerControl.setFocus();
			}
			// types are originals
			member = JavaModelUtil.toOriginal(member);

			if (!member.equals(fSelectedType)) {
				getCurrentViewer().setSelection(
					new StructuredSelection(member),
					true);
			}
		}
	}

	/**
	 * @deprecated
	 */
	public IType getInput() {
		if (fInputElement instanceof IType) {
			return (IType) fInputElement;
		}
		return null;
	}

	/**
	 * Sets the input to a new type
	 * @deprecated 
	 */
	public void setInput(IType type) {
		setInputElement(type);
	}

	/**
	 * Returns the input element of the type hierarchy.
	 * Can be of type <code>IType</code> or <code>IPackageFragment</code>
	 */
	public IJavaElement getInputElement() {
		return fInputElement;
	}

	/**
	 * Sets the input to a new element.
	 */
	public void setInputElement(IJavaElement element) {
		if (element != null) {
			if (element instanceof IMember) {
				if (element.getElementType() != IJavaElement.TYPE) {
					element = ((IMember) element).getDeclaringType();
				}
				ICompilationUnit cu = ((IMember) element).getCompilationUnit();
				if (cu != null && cu.isWorkingCopy()) {
					element.getPrimaryElement(); 
					//cu.getOriginal(element);
					if (!element.exists()) {
						MessageDialog.openError(getSite().getShell(), TypeHierarchyMessages.getString("TypeHierarchyViewPart.error.title"), TypeHierarchyMessages.getString("TypeHierarchyViewPart.error.message")); //$NON-NLS-1$ //$NON-NLS-2$
						return;
					}
				}
			} else {
				int kind = element.getElementType();
				if (kind != IJavaElement.JAVA_PROJECT
					&& kind != IJavaElement.PACKAGE_FRAGMENT_ROOT
					&& kind != IJavaElement.PACKAGE_FRAGMENT) {
					element = null;
					JavaPlugin.logErrorMessage("Invalid type hierarchy input type."); //$NON-NLS-1$
				}
			}
		}

		updateInput(element);
	}

	/**
	 * Changes the input to a new type
	 */
	private void updateInput(IJavaElement inputElement) {
		IJavaElement prevInput = fInputElement;

		// Make sure the UI got repainted before we execute a long running
		// operation. This can be removed if we refresh the hierarchy in a 
		// separate thread.
		// Work-araound for http://dev.eclipse.org/bugs/show_bug.cgi?id=30881
		processOutstandingEvents();
		if (inputElement == null) {
			clearInput();
		} else {
			fInputElement = inputElement;
			try {
				fHierarchyLifeCycle.ensureRefreshedTypeHierarchy(
					inputElement,
					JavaPlugin.getActiveWorkbenchWindow());
				// fHierarchyLifeCycle.ensureRefreshedTypeHierarchy(inputElement, getSite().getWorkbenchWindow());
			} catch (InvocationTargetException e) {
				ExceptionHandler.handle(e, getSite().getShell(), TypeHierarchyMessages.getString("TypeHierarchyViewPart.exception.title"), TypeHierarchyMessages.getString("TypeHierarchyViewPart.exception.message")); //$NON-NLS-1$ //$NON-NLS-2$
				clearInput();
				return;
			} catch (InterruptedException e) {
				return;
			}

			if (inputElement.getElementType() != IJavaElement.TYPE) {
				setView(VIEW_ID_TYPE);
			}
			// turn off member filtering

			if (!inputElement.equals(prevInput)) {
				updateHierarchyViewer(true);
			}
			IType root = getSelectableType(inputElement);
			internalSelectType(root, true);

			updateTitle();
			fPagebook.showPage(fTypeMethodsSplitter);
		}
	}

	private void processOutstandingEvents() {
		Display display = getDisplay();
		if (display != null && !display.isDisposed())
			display.update();
	}

	private void clearInput() {
		fInputElement = null;
		fHierarchyLifeCycle.freeHierarchy();

		updateHierarchyViewer(false);

	}

	/*
	 * @see IWorbenchPart#setFocus
	 */
	public void setFocus() {
		fPagebook.setFocus();
	}

	/*
	 * @see IWorkbenchPart#dispose
	 */
	public void dispose() {
		fHierarchyLifeCycle.freeHierarchy();
		fHierarchyLifeCycle.removeChangedListener(
			fTypeHierarchyLifeCycleListener);
		fPaneLabelProvider.dispose();

		getSite().getPage().removePartListener(fPartListener);

		if (fActionGroups != null)
			fActionGroups.dispose();
		super.dispose();
	}

	/**
	 * Answer the property defined by key.
	 */
	public Object getAdapter(Class key) {
		if (key == IShowInSource.class) {
			return getShowInSource();
		}
		if (key == IShowInTargetList.class) {
			return new IShowInTargetList() {
				public String[] getShowInTargetIds() {
					return new String[] {
						JavaUI.ID_PACKAGES,
						IPageLayout.ID_RES_NAV };
				}

			};
		}
		return super.getAdapter(key);
	}

	private Control createTypeViewerControl(Composite parent) {
		fViewerbook = new PageBook(parent, SWT.NULL);

		KeyListener keyListener = createKeyListener();
		AmbientHierarchyViewer vajViewer =
			new AmbientTraditionalHierarchyViewer(
				fViewerbook,
				fHierarchyLifeCycle,
				this);
		initializeTypesViewer(
			vajViewer,
			keyListener,
			IContextMenuConstants.TARGET_ID_HIERARCHY_VIEW);

		fAllViewers = new AmbientHierarchyViewer[1];
		fAllViewers[VIEW_ID_TYPE] = vajViewer;

		int currViewerIndex;
		currViewerIndex = VIEW_ID_TYPE;

		fEmptyTypesViewer = new Label(fViewerbook, SWT.LEFT);

		for (int i = 0; i < fAllViewers.length; i++) {
			fAllViewers[i].setInput(fAllViewers[i]);
		}

		// force the update
		fCurrentViewerIndex = -1;
		setView(currViewerIndex);

		return fViewerbook;
	}

	private KeyListener createKeyListener() {
		return new KeyAdapter() {
			public void keyReleased(KeyEvent event) {
				if (event.stateMask == 0) {
					if (event.keyCode == SWT.F5) {
						updateHierarchyViewer(false);
						return;
					} else if (event.character == SWT.DEL) {
						if (fCCPActionGroup.getDeleteAction().isEnabled())
							fCCPActionGroup.getDeleteAction().run();
						return;
					}
				}
			}
		};
	}

	private void initializeTypesViewer(
		final AmbientHierarchyViewer typesViewer,
		KeyListener keyListener,
		String cotextHelpId) {
		typesViewer.getControl().setVisible(false);
		typesViewer.getControl().addKeyListener(keyListener);
		typesViewer.initContextMenu(new IMenuListener() {
			public void menuAboutToShow(IMenuManager menu) {
				fillTypesViewerContextMenu(typesViewer, menu);
			}
		}, cotextHelpId, getSite());
		typesViewer.addSelectionChangedListener(fSelectionChangedListener);

	}

	private void initDragAndDrop() {
		Transfer[] transfers =
			new Transfer[] { LocalSelectionTransfer.getInstance()};
		int ops = DND.DROP_MOVE | DND.DROP_COPY | DND.DROP_LINK;

		for (int i = 0; i < fAllViewers.length; i++) {
			addDragAdapters(fAllViewers[i], ops, transfers);
			addDropAdapters(fAllViewers[i], ops | DND.DROP_DEFAULT, transfers);
		}

		//dnd on empty hierarchy
		DropTarget dropTarget =
			new DropTarget(fNoHierarchyShownLabel, ops | DND.DROP_DEFAULT);
		dropTarget.setTransfer(transfers);
		dropTarget.addDropListener(
			new AmbientHierarchyDragAndDropAdapter(this, fAllViewers[0]));
	}

	private void addDropAdapters(
		AbstractTreeViewer viewer,
		int ops,
		Transfer[] transfers) {
		TransferDropTargetListener[] dropListeners =
			new TransferDropTargetListener[] {
				 new AmbientHierarchyDragAndDropAdapter(this, viewer)};
		viewer.addDropSupport(
			ops,
			transfers,
			new DelegatingDropAdapter(dropListeners));
	}

	private void addDragAdapters(
		StructuredViewer viewer,
		int ops,
		Transfer[] transfers) {
		TransferDragSourceListener[] dragListeners =
			new TransferDragSourceListener[] {
				 new SelectionTransferDragAdapter(viewer)};
		viewer.addDragSupport(
			ops,
			transfers,
			new JdtViewerDragAdapter(viewer, dragListeners));
	}

	/**
	 * Returns the inner component in a workbench part.
	 * @see IWorkbenchPart#createPartControl
	 */
	public void createPartControl(Composite container) {

		fPagebook = new PageBook(container, SWT.NONE);

		// page 1 of pagebook (viewers)

		fTypeMethodsSplitter = new SashForm(fPagebook, SWT.VERTICAL);
		fTypeMethodsSplitter.setVisible(false);

		fTypeViewerViewForm = new ViewForm(fTypeMethodsSplitter, SWT.NONE);

		Control typeViewerControl =
			createTypeViewerControl(fTypeViewerViewForm);
		fTypeViewerViewForm.setContent(typeViewerControl);

		// page 2 of pagebook (no hierarchy label)
		fNoHierarchyShownLabel =
			new Label(fPagebook, SWT.TOP + SWT.LEFT + SWT.WRAP);
		fNoHierarchyShownLabel.setText(
			"Please select a project, or an element of a project to enable the hierarchy view.");

		MenuManager menu = new MenuManager();
		fNoHierarchyShownLabel.setMenu(
			menu.createContextMenu(fNoHierarchyShownLabel));

		fPagebook.showPage(fNoHierarchyShownLabel);

		// selection provider
		int nHierarchyViewers = fAllViewers.length;
		StructuredViewer[] trackedViewers = new StructuredViewer[nHierarchyViewers];
		for (int i = 0; i < nHierarchyViewers; i++) {
			trackedViewers[i] = (StructuredViewer)fAllViewers[i];
		}
		fSelectionProviderMediator =
			new SelectionProviderMediator(trackedViewers);
		
		//TODO
		IStatusLineManager slManager =
			getViewSite().getActionBars().getStatusLineManager();
		fSelectionProviderMediator.addSelectionChangedListener(
			new StatusBarUpdater(slManager));

		getSite().setSelectionProvider(fSelectionProviderMediator);
		getSite().getPage().addPartListener(fPartListener);

		// see http://bugs.eclipse.org/bugs/show_bug.cgi?id=33657
		IJavaElement input = null; //determineInputElement();
		if (input != null) {
			setInputElement(input);
		} else {
			setViewerVisibility(false);
		}

		WorkbenchHelp.setHelp(
			fPagebook,
			IJavaHelpContextIds.TYPE_HIERARCHY_VIEW);

		fActionGroups =
			new CompositeActionGroup(
				new ActionGroup[] {
					new NewWizardsActionGroup(this.getSite()),
					new OpenEditorActionGroup(this),
					new OpenViewActionGroup(this),
					fCCPActionGroup = new CCPActionGroup(this),
					new GenerateActionGroup(this),
					new RefactorActionGroup(this),
					new JavaSearchActionGroup(this)});

		initDragAndDrop();
	}

	/**
	 * Creates the context menu for the hierarchy viewers
	 */
	private void fillTypesViewerContextMenu(
		AmbientHierarchyViewer viewer,
		IMenuManager menu) {
		JavaPlugin.createStandardGroups(menu);

		menu.appendToGroup(
			IContextMenuConstants.GROUP_SHOW,
			new Separator(GROUP_FOCUS));
		// myViewer entries
		viewer.contributeToContextMenu(menu);

		fActionGroups.setContext(
			new ActionContext(getSite().getSelectionProvider().getSelection()));
		fActionGroups.fillContextMenu(menu);
		fActionGroups.setContext(null);
	}

	/**
	 * Toggles between the empty myViewer page and the hierarchy
	 */
	private void setViewerVisibility(boolean showHierarchy) {
		if (showHierarchy) {
			fViewerbook.showPage(getCurrentViewer().getControl());
		} else {
			fViewerbook.showPage(fEmptyTypesViewer);
		}
	}

	private IType getSelectableType(IJavaElement elem) {
		if (elem.getElementType() != IJavaElement.TYPE) {
			return null; //(IType) getCurrentViewer().getTreeRootType();
		} else {
			return (IType) elem;
		}
	}

	private void internalSelectType(IMember elem, boolean reveal) {
		AmbientHierarchyViewer viewer = getCurrentViewer();
		viewer.removeSelectionChangedListener(fSelectionChangedListener);
		viewer.setSelection(
			elem != null
				? new StructuredSelection(elem)
				: StructuredSelection.EMPTY,
			reveal);
		viewer.addSelectionChangedListener(fSelectionChangedListener);
	}

	/**
	 * When the input changed or the hierarchy pane becomes visible,
	 * <code>updateHierarchyViewer<code> brings up the correct view and refreshes
	 * the current tree
	 */
	private void updateHierarchyViewer(final boolean doExpand) {
		if (fInputElement == null) {
			fPagebook.showPage(fNoHierarchyShownLabel);
		} else {
			if (getCurrentViewer().containsElements() != null) {
				Runnable runnable = new Runnable() {
					public void run() {
						getCurrentViewer().updateContent(doExpand); // refresh
					}
				};
				BusyIndicator.showWhile(getDisplay(), runnable);
				if (!isChildVisible(fViewerbook,
					getCurrentViewer().getControl())) {
					setViewerVisibility(true);
				}
			} else {
				fEmptyTypesViewer.setText(TypeHierarchyMessages.getFormattedString("TypeHierarchyViewPart.nodecl", fInputElement.getElementName())); //$NON-NLS-1$
				setViewerVisibility(false);
			}
		}
	}

	protected void doSelectionChanged(SelectionChangedEvent e) {
		typeSelectionChanged(e.getSelection());
	}

	private void typeSelectionChanged(ISelection sel) {
		if (sel instanceof IStructuredSelection) {
			List selected = ((IStructuredSelection) sel).toList();
			int nSelected = selected.size();
			if (nSelected != 0) {
				List types = new ArrayList(nSelected);
				for (int i = nSelected - 1; i >= 0; i--) {
					Object elem = selected.get(i);
					if (elem instanceof IType && !types.contains(elem)) {
						types.add(elem);
					}
				}
				if (types.size() == 1) {
					fSelectedType = (IType) types.get(0);

				} else if (types.size() == 0) {
					// method selected, no change
				}
				if (nSelected == 1) {
					revealElementInEditor(selected.get(0), getCurrentViewer());
				}
			} else {
				fSelectedType = null;

			}
		}
	}

	private void revealElementInEditor(Object elem, Viewer originViewer) {
		// only allow revealing when the type hierarchy is the active pagae
		// no revealing after selection events due to model changes

		if (getSite().getPage().getActivePart() != this) {
			return;
		}

		if (fSelectionProviderMediator.getViewerInFocus() != originViewer) {
			return;
		}

		IEditorPart editorPart = EditorUtility.isOpenInEditor(elem);
		if (editorPart != null && (elem instanceof IJavaElement)) {
			getSite().getPage().removePartListener(fPartListener);
			getSite().getPage().bringToTop(editorPart);
			EditorUtility.revealInEditor(editorPart, (IJavaElement) elem);
			getSite().getPage().addPartListener(fPartListener);
		}
	}

	private Display getDisplay() {
		if (fPagebook != null && !fPagebook.isDisposed()) {
			return fPagebook.getDisplay();
		}
		return null;
	}

	private boolean isChildVisible(Composite pb, Control child) {
		Control[] children = pb.getChildren();
		for (int i = 0; i < children.length; i++) {
			if (children[i] == child && children[i].isVisible())
				return true;
		}
		return false;
	}

	private void updateTitle() {
		String viewerTitle = getCurrentViewer().getTitle();

		String tooltip;
		String title;
		if (fInputElement != null) {
			String[] args =
				new String[] {
					viewerTitle,
					JavaElementLabels.getElementLabel(
						fInputElement,
						JavaElementLabels.ALL_DEFAULT)};
			title = TypeHierarchyMessages.getFormattedString("TypeHierarchyViewPart.title", args); //$NON-NLS-1$
			tooltip = TypeHierarchyMessages.getFormattedString("TypeHierarchyViewPart.tooltip", args); //$NON-NLS-1$
		} else {
			title = viewerTitle;
			tooltip = viewerTitle;
		}
		setPartName(title);
		setTitleToolTip(tooltip);
	}

	/**
	 * Sets the current view (see view id)
	 * called from ToggleViewAction. Must be called after creation of the viewpart.
	 */
	public void setView(int viewerIndex) {
		Assert.isNotNull(fAllViewers);
		if (viewerIndex < fAllViewers.length
			&& fCurrentViewerIndex != viewerIndex) {
			fCurrentViewerIndex = viewerIndex;

			updateHierarchyViewer(true);
			if (fInputElement != null) {
				ISelection currSelection = getCurrentViewer().getSelection();
				if (currSelection == null || currSelection.isEmpty()) {
					internalSelectType(getSelectableType(fInputElement), false);
					currSelection = getCurrentViewer().getSelection();
				}

			}
			updateTitle();

			fDialogSettings.put(DIALOGSTORE_HIERARCHYVIEW, viewerIndex);
			getCurrentViewer().getTree().setFocus();
		}

	}

	/**
	 * Gets the curret active view index.
	 */
	public int getViewIndex() {
		return fCurrentViewerIndex;
	}

	private AmbientHierarchyViewer getCurrentViewer() {
		return fAllViewers[fCurrentViewerIndex];
	}

	/**
	 * Called from ITypeHierarchyLifeCycleListener.
	 * Can be called from any thread
	 */
	protected void doTypeHierarchyChanged(
		final TypeHierarchyLifeCycle typeHierarchy,
		final IType[] changedTypes) {
		if (!fIsVisible) {
			fNeedRefresh = true;
			return;
		}
		if (fIsRefreshRunnablePosted) {
			return;
		}

		Display display = getDisplay();
		if (display != null) {
			fIsRefreshRunnablePosted = true;
			display.asyncExec(new Runnable() {
				public void run() {
					try {
						if (fPagebook != null && !fPagebook.isDisposed()) {
							doTypeHierarchyChangedOnViewers(changedTypes);
						}
					} finally {
						fIsRefreshRunnablePosted = false;
					}
				}
			});
		}
	}

	protected void doTypeHierarchyChangedOnViewers(IType[] changedTypes) {
		if (fHierarchyLifeCycle.getHierarchy() == null
			|| !fHierarchyLifeCycle.getHierarchy().exists()) {
			clearInput();
		} else {
			if (changedTypes == null) {
				// hierarchy change
				try {
					fHierarchyLifeCycle.ensureRefreshedTypeHierarchy(
						fInputElement,
						getSite().getWorkbenchWindow());
				} catch (InvocationTargetException e) {
					ExceptionHandler.handle(e, getSite().getShell(), TypeHierarchyMessages.getString("TypeHierarchyViewPart.exception.title"), TypeHierarchyMessages.getString("TypeHierarchyViewPart.exception.message")); //$NON-NLS-1$ //$NON-NLS-2$
					clearInput();
					return;
				} catch (InterruptedException e) {
					return;
				}

				updateHierarchyViewer(false);
			} else {
				// elements in hierarchy modified

				if (getCurrentViewer().isMethodFiltering()) {
					if (changedTypes.length == 1) {
						getCurrentViewer().refresh(changedTypes[0]);
					} else {
						updateHierarchyViewer(false);
					}
				} else {
					getCurrentViewer().update(
						changedTypes,
						new String[] {
							IBasicPropertyConstants.P_TEXT,
							IBasicPropertyConstants.P_IMAGE });
				}
			}
		}
	}

	/*
	 * @see IViewPart#init
	 */
	public void init(IViewSite site, IMemento memento)
		throws PartInitException {
		super.init(site, memento);

	}

	/**
	 * view part becomes visible
	 */
	protected void visibilityChanged(boolean isVisible) {
		fIsVisible = isVisible;
		if (isVisible && fNeedRefresh) {
			doTypeHierarchyChanged(fHierarchyLifeCycle, null);
		}
		fNeedRefresh = false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jdt.internal.ui.viewsupport.IViewPartInputProvider#getViewPartInput()
	 */
	public Object getViewPartInput() {
		return fInputElement;
	}

	/**
	 * Returns the <code>IShowInSource</code> for this view.
	 */
	protected IShowInSource getShowInSource() {
		return new IShowInSource() {
			public ShowInContext getShowInContext() {
				return new ShowInContext(
					null,
					getSite().getSelectionProvider().getSelection());
			}
		};
	}

}
