/*******************************************************************************
 * Copyright (c) 2004 Duke University
 *
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.cs.duke.edu/csed/ambient/copyright.html
 * 
*******************************************************************************/
/*******************************************************************************
 * This class is edited from the original TraditionalHierarchyViewer in
 * org.eclipse.jdt.internal.ui.typehierarchy
 *******************************************************************************/
package edu.duke.ambient.ui.hierarchy;

import org.eclipse.jdt.core.Flags;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.ITypeHierarchy;
import org.eclipse.jdt.internal.corext.util.JavaModelUtil;
import org.eclipse.jdt.internal.ui.typehierarchy.TypeHierarchyContentProvider;
import org.eclipse.jdt.internal.ui.typehierarchy.TypeHierarchyLifeCycle;
import org.eclipse.jdt.internal.ui.typehierarchy.TypeHierarchyMessages;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IWorkbenchPart;

/**
 * A TypeHierarchyViewer that looks like the type hierarchy view of VA/Java:
 * Starting form Object down to the element in focus, then all subclasses from
 * this element.
 * Used by the TypeHierarchyViewPart which has to provide a TypeHierarchyLifeCycle
 * on construction (shared type hierarchy)
 */
public class AmbientTraditionalHierarchyViewer extends AmbientHierarchyViewer {
	
	public AmbientTraditionalHierarchyViewer(Composite parent, TypeHierarchyLifeCycle lifeCycle, IWorkbenchPart part) {
		super(parent, new AmbientTraditionalHierarchyContentProvider(lifeCycle), lifeCycle, part);
	}
	
	/*
	 * @see TypeHierarchyViewer#getTitle
	 */	
	public String getTitle() {
		if (isMethodFiltering()) {
			return TypeHierarchyMessages.getString("TraditionalHierarchyViewer.filtered.title"); //$NON-NLS-1$
		} else {
			return TypeHierarchyMessages.getString("TraditionalHierarchyViewer.title"); //$NON-NLS-1$
		}
	}

	/*
	 * @see TypeHierarchyViewer#updateContent
	 */		
	public void updateContent(boolean expand) {
		getTree().setRedraw(false);
		refresh();
		
		if (expand) {
			AmbientTraditionalHierarchyContentProvider contentProvider= (AmbientTraditionalHierarchyContentProvider) getContentProvider();
			int expandLevel= contentProvider.getExpandLevel();
			if (isMethodFiltering()) {
				expandLevel++;
			}
			expandToLevel(expandLevel);
		}
		getTree().setRedraw(true);
	}	

	/**
	 * Content provider for the 'traditional' type hierarchy.
	 */	
	private static class AmbientTraditionalHierarchyContentProvider extends TypeHierarchyContentProvider 
	{
		//protected void getTypesInHierarchy(IType t, List l)
		//{}
		
		public AmbientTraditionalHierarchyContentProvider(TypeHierarchyLifeCycle provider) {
			super(provider);
		}
		
		public int getExpandLevel() {
			ITypeHierarchy hierarchy= getHierarchy();
			if (hierarchy != null) {
				IType input= hierarchy.getType();
				if (input != null) {
					return getDepth(hierarchy, input) + 2;
				} else {
					return 5;
				}
			}
			return 2;
		}
		
		private int getDepth(ITypeHierarchy hierarchy, IType input) {
			int count= 0;
			IType superType= hierarchy.getSuperclass(input);
			while (superType != null) {
				count++;
				superType= hierarchy.getSuperclass(superType);
			}
			return count;
		}
		
	
		/*
		 * @see TypeHierarchyContentProvider.getElements
		 */
		public Object[] getElements(Object parent) {
			
			ITypeHierarchy hierarchy= getHierarchy();
			if (hierarchy != null) {
				IType input= hierarchy.getType();
				if (input == null) {
					// opened on a region
					return hierarchy.getRootClasses();	
				} else {
					if (Flags.isInterface(hierarchy.getCachedFlags(input))) {
						return new Object[] { input };
					} else {
						IType[] roots= hierarchy.getRootClasses();
						for (int i= 0; i < roots.length; i++) {
							if ("java.lang.Object".equals(JavaModelUtil.getFullyQualifiedName(roots[i]))) { //$NON-NLS-1$
								return new Object[] { roots[i] };
							}
						} 
						return roots; // a problem with the hierarchy
					}
				}
			}
			return NO_ELEMENTS;
		}
	
		/*
		 * @see TypeHierarchyContentProvider.getTypesInHierarchy
		 */	
		
		protected IType getParentType(IType type) {
			ITypeHierarchy hierarchy= getHierarchy();
			if (hierarchy != null) {
				return hierarchy.getSuperclass(type);
				// dont handle interfaces
			}
			return null;
		}	
			
		/* (non-Javadoc)
		 * @see org.eclipse.jdt.internal.ui.typehierarchy.TypeHierarchyContentProvider#getTypesInHierarchy(org.eclipse.jdt.core.IType, java.util.List)
		 */
		protected void getTypesInHierarchy(IType type, java.util.List res) 
		{
				ITypeHierarchy hierarchy= getHierarchy();
				if (hierarchy != null) {
					//we want to show all interfaces we created
					if(type.getElementName().equals("Object")){
					    IType[] interfaceArray = hierarchy.getRootInterfaces();
						IType[] classArray = hierarchy.getSubtypes(type);
						IType[] returnArray = new IType[interfaceArray.length + classArray.length];
						for (int i = 0; i < interfaceArray.length; i++){
							returnArray[i] = interfaceArray[i];
						}
						for (int i = interfaceArray.length; i < interfaceArray.length + classArray.length; i++){
							returnArray[i] = classArray[i-interfaceArray.length];
						}
						//return returnArray; 
						for (int i=0; i<returnArray.length;i++)
							res.add(returnArray[i].toString());
						return;
					}
					//return hierarchy.getSubtypes(type);
					for (int i=0; i<hierarchy.getSubtypes(type).length;i++)
						res.add(hierarchy.getSubtypes(type)[i].toString());
					return;
				}
				return;
			}
		}
}

