package edu.duke.cs.snarfer;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.FactoryConfigurationError;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import edu.duke.cs.snarfer.PackageEntry;
import edu.duke.cs.snarfer.PackageException;
import edu.duke.cs.snarfer.PackageInfo;
import edu.duke.cs.util.Util;

/**
 * 
 * @author Ethan Eade
 *	Package holds and handles the information pertaining to a package descriptor xml file.
 * @see PackageInfo, PackageSite
 */
public class Package
{
    private String myDescription;
    private PackageInfo myInfo;
    private ArrayList myEntries;

    /**
     * Build a Package object given a PackageInfo, a collection of PackageEntry's, and a description.
     * @param info contains the details of the package
     * @param entries a collection of PackageEntry objects to add to the package
     * @param desc the description of the package
     */
    public Package (PackageInfo info, Collection entries, String desc)
    {
        myInfo = info;
        myEntries = new ArrayList(entries);
        myDescription = desc;
    }

    /**
     * Construct a package by contacting the URL given by the PackageInfo, and parsing the description in that XML file.
     * @param info the PackageInfo containing a valid URL for a package description XML file.
     * @throws PackageException
     */
    public Package (PackageInfo info) throws PackageException
    {
        myInfo = info;
        Document doc;
        try
	{
            DocumentBuilder builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
            doc = builder.parse(myInfo.getURL().openStream());
        }
        catch (ParserConfigurationException e)
	{
            throw new PackageException("Could not create XML parser", e);
        }
        catch (FactoryConfigurationError e)
	{
            throw new PackageException("Could not create XML parser",e);
        }
        catch (SAXException e)
	{
            throw new PackageException("Could not parse XML", e);
        }
        catch (IOException e)
	{
            throw new PackageException("Error fetching descriptor", e);
        }
        Element root = doc.getDocumentElement();
        buildFrom(root);
    }

    protected void buildFrom (Element root) throws PackageException
    {
        if (!root.getNodeName().equals("package"))
            throw new PackageException("Expected 'package' as root tag");
        if (root.hasAttribute("version"))
            myInfo.setVersion(root.getAttribute("version"));
        if (root.hasAttribute("name"))
            myInfo.setName(root.getAttribute("name"));
        if (root.hasAttribute("publisher"))
            myInfo.setPublisher(root.getAttribute("publisher"));
        NodeList descs = root.getElementsByTagName("description");
        Node desc = descs.item(0);
        myDescription = desc.getFirstChild().getNodeValue();
        NodeList files = root.getElementsByTagName("file");
        myEntries = new ArrayList();
        for (int i = 0;i<files.getLength();i++)
	{
            Element f = (Element)files.item(i);
            try
	    {
                URL url = new URL(myInfo.getURL(), f.getAttribute("url"));
                String archive = f.getAttribute("archive");
                myEntries.add(new PackageEntry(url, archive.equals("true")||archive.equals("yes")));
            }
            catch (MalformedURLException e)
	    {
                throw new PackageException("Bad file URL", e);
            }
        }
    }

    /**
     * Build a DOM document, representing the XML of this Package.
     * @return a DOM Document that represents the Package in XML.
     */
    public Document toDOM ()
    {
        Document doc = Util.createDOM();
        Element root = doc.createElement("package");
        root.setAttribute("name", myInfo.getName());
        root.setAttribute("version", myInfo.getVersion());
        root.setAttribute("publisher", myInfo.getPublisher());
        root.setAttribute("type", myInfo.getType());
        Element desc = doc.createElement("description");
        desc.appendChild(doc.createTextNode(myDescription));
        root.appendChild(desc);
        for (int i = 0; i < myEntries.size(); i++)
	{
	    PackageEntry entry = (PackageEntry)myEntries.get(i);
            Element el = doc.createElement("file");
            el.setAttribute("url",entry.getURL().toString());
            el.setAttribute("archive", entry.isArchive() ? "true" : "false");
            root.appendChild(el);
        }
        doc.appendChild(root);
        return doc;
    }

    /**
     * @return the number of entries in this package
     */
    public int getEntryCount ()
    {
        return myEntries.size();
    }

    /**
     * Get the i'th PackageEntry.
     * @param i which entry to return
     * @return the i'th PackageEntry
     */
    public PackageEntry getEntry (int i)
    {
        if (0 <= i && i < myEntries.size())
	{
	    return (PackageEntry)myEntries.get(i);
	}
	return null;
    }

    public Collection getEntries ()
    {
        return myEntries;
    }

    public PackageInfo getInfo ()
    {
        return myInfo;
    }

    public String getDescription ()
    {
        return myDescription;
    }
}
