package edu.duke.cs.snarfer.eclipse;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Iterator;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.GroupMarker;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.part.ViewPart;

import edu.duke.cs.snarfer.PackageException;
import edu.duke.cs.snarfer.PackageInfo;
import edu.duke.cs.snarfer.Package;
/**
 * @see ViewPart
 */
public class SiteViewPart extends ViewPart implements ISelectionChangedListener, SelectionListener {
	private SiteContentProvider contentProvider;
	private SiteLabelProvider labelProvider;
	private Composite panel;
	private TreeViewer viewer;
	private Label statusLabel;
	private Button refreshButton;
	private NewSiteAction newSiteAction;
	private RemoveSiteAction removeSiteAction;
	private OpenAction openAction;
	private InstallAction installAction;
	private RefreshAction refreshAction;
	private EditSiteAction editSiteAction;
	private UpdateSearchAction updateSearchAction;
	public SiteViewPart() {
		super();
				
		contentProvider = new SiteContentProvider();
		labelProvider = new SiteLabelProvider();
		newSiteAction = new NewSiteAction(Strings.getString("SiteViewPart.new.text")); //$NON-NLS-1$
		newSiteAction.setToolTipText(Strings.getString("SiteViewPart.new.tooltip")); //$NON-NLS-1$
		newSiteAction.setImageDescriptor(SnarferPlugin.getImageDescriptor("icons/install.gif"));
		editSiteAction = new EditSiteAction();
		editSiteAction.setImageDescriptor(PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(ISharedImages.IMG_OBJS_INFO_TSK));
		editSiteAction.setText(Strings.getString("SiteViewPart.edit.text")); //$NON-NLS-1$
		editSiteAction.setToolTipText(Strings.getString("SiteViewPart.edit.tooltip")); //$NON-NLS-1$
		removeSiteAction = new RemoveSiteAction();
		removeSiteAction.setImageDescriptor(PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(ISharedImages.IMG_TOOL_DELETE));
		openAction = new OpenAction();
		openAction.setImageDescriptor(PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(ISharedImages.IMG_OBJ_FOLDER)); //$NON-NLS-1$
		installAction = new InstallAction();
		installAction.setImageDescriptor(PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(IDE.SharedImages.IMG_OBJ_PROJECT)); //$NON-NLS-1$
		refreshAction = new RefreshAction();
		refreshAction.setText(Strings.getString("SiteViewPart.refresh.text")); //$NON-NLS-1$
		refreshAction.setToolTipText(Strings.getString("SiteViewPart.refresh.tooltip")); //$NON-NLS-1$
		refreshAction.setImageDescriptor(SnarferPlugin.getImageDescriptor("icons/refresh.gif")); //$NON-NLS-1$
		updateSearchAction = new UpdateSearchAction();
		updateSearchAction.setToolTipText(Strings.getString("SiteViewPart.update.tooltip")); //$NON-NLS-1$
		updateSearchAction.setImageDescriptor(SnarferPlugin.getImageDescriptor("icons/search.gif")); //$NON-NLS-1$
		removeSiteAction.setEnabled(false);
		openAction.setEnabled(false);
		editSiteAction.setEnabled(false);
		installAction.setEnabled(false);
	}

	/**
	 * @see ViewPart#createPartControl
	 */
	
	public void createPartControl(Composite parent)  {
		parent.setLayout(new FillLayout());	
		panel = new Composite(parent, SWT.NONE);
		panel.setLayout(new GridLayout(1,false));
		statusLabel = new Label(panel, SWT.HORIZONTAL);
		statusLabel.setText(Strings.getString("SiteViewPart.status.sites")); //$NON-NLS-1$
		viewer = new TreeViewer(panel);
		viewer.setContentProvider(contentProvider);
		viewer.setLabelProvider(labelProvider);
		viewer.getControl().setLayoutData(new GridData(GridData.FILL_BOTH));
		viewer.addSelectionChangedListener(this);
		viewer.getTree().addSelectionListener(this);
		createContextMenu();
		createToolbar();
		onRefresh();
	}

	public void dispose() {
		super.dispose();
		statusLabel.dispose();
		panel.dispose();
	}
	/**
	 * @see ViewPart#setFocus
	 */
	public void setFocus()  {
		viewer.getControl().setFocus();
	}
	private void onRefresh() {
		Iterator it = SnarferPlugin.getDefault().getSiteURLs().iterator();
		while (it.hasNext()) {
			SnarferPlugin.getDefault().uncacheSite((URL)it.next());
		}
		viewer.setInput(SnarferPlugin.getDefault().getSiteURLs());
		viewer.expandAll();		
	}
	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	private class NewSiteAction extends Action {
		public NewSiteAction(String name) {
			super(name);
		}
		public void run() {
			NewSiteDialog d = new NewSiteDialog(viewer, getSite().getShell());
			if (d.open() == Dialog.CANCEL)
				return;
			SnarferPlugin.getDefault().getSiteURLs().add(d.getURL());
			try {
				SnarferPlugin.getDefault().saveSettings();
			}
			catch (IOException e) {
				System.err.println("Error saving plugin settings.");
			}
			SnarferPlugin.getDefault().uncacheSite(d.getURL());
			viewer.setInput(SnarferPlugin.getDefault().getSiteURLs());
			viewer.refresh();
		}
	}
	private class EditSiteAction extends Action implements IInputValidator {
		private URL url;
		public EditSiteAction() {
			super("Edit site"); //$NON-NLS-1$
		}
		public void setURL(URL url) {
			this.url = url;
			setText(Strings.getString("SiteViewPart.edit.urlprefix")+url+"'"); //$NON-NLS-1$ //$NON-NLS-2$
			setToolTipText(getText());
		}
		public void run() {
			EditSiteDialog dlg = new EditSiteDialog(viewer, getSite().getShell(),url);
			if (dlg.open() == Dialog.CANCEL)
				return;
			try {
				SnarferPlugin.getDefault().getSiteURLs().remove(url);
				url = new URL(dlg.getValue());
				SnarferPlugin.getDefault().getSiteURLs().add(url);
				SnarferPlugin.getDefault().saveSettings();
			}
			catch (MalformedURLException e) {
				return;
			}
			catch (IOException e) {
				e.printStackTrace();
			}
			viewer.setInput(SnarferPlugin.getDefault().getSiteURLs());
		}
		/* (non-Javadoc)
		 * @see org.eclipse.jface.dialogs.IInputValidator#isValid(java.lang.String)
		 */
		public String isValid(String newText) {
			try {
				URL test = new URL(newText);
				return null;
			}
			catch (java.net.MalformedURLException e) {
				return Strings.getString("SiteViewPart.edit.dlg.invalid"); //$NON-NLS-1$
			}
		}

	}
	private class RemoveSiteAction extends Action {
		private URL url;
		public RemoveSiteAction() {
			super(Strings.getString("SiteViewPart.remove.text")); //$NON-NLS-1$
		}
		public void setURL(URL url) {
			this.url = url;
			setText(Strings.getString("SiteViewPart.remove.urlprefix")+url+"'"); //$NON-NLS-1$ //$NON-NLS-2$
			setToolTipText(getText());
		}
		public void run() {
			if (!MessageDialog.openQuestion(getSite().getShell(), 
				Strings.getString("SiteViewPart.remove.confirm.title"),  //$NON-NLS-1$
				Strings.getString("SiteViewPart.remove.confirm.prefix")+url+"'?")) //$NON-NLS-1$ //$NON-NLS-2$
				return;
			SnarferPlugin.getDefault().getSiteURLs().remove(url);
			try {
				SnarferPlugin.getDefault().saveSettings();
			}
			catch (IOException e) {
				e.printStackTrace();
			}
			viewer.setInput(SnarferPlugin.getDefault().getSiteURLs());
		}
	}
	private class InstallAction extends Action {
		PackageInfo info;
		public InstallAction() {
			super("Open"); //$NON-NLS-1$
		}
		public void setPackageInfo(PackageInfo info) {
			setText(Strings.getString("SiteViewPart.install.pkgprefix")+info.getName()+" "+info.getVersion()); //$NON-NLS-1$ //$NON-NLS-2$
			setToolTipText(getText());
			this.info = info;
		}
		public void run() {
				Package pkg = null;
				try {
					pkg = new Package(info);
				} catch (PackageException e) {
					MessageDialog.openError(getSite().getShell(),Strings.getString("InstallWizard.load.error"), "Unable to open project site!");
				}
				InstallWizard installWiz = new InstallWizard(pkg, SnarferPlugin.getDefault().getLastInstallLocation());
				WizardDialog dlg = new WizardDialog(getSite().getShell(), installWiz);
				dlg.open();
		}
	}
	private class OpenAction extends Action {
		PackageInfo info;
		public OpenAction() {
			super("Open"); //$NON-NLS-1$
		}
		public void setPackageInfo(PackageInfo info) {
			setText(Strings.getString("SiteViewPart.open.pkgprefix")+info.getName()+" "+info.getVersion()); //$NON-NLS-1$ //$NON-NLS-2$
			setToolTipText(getText());
			this.info = info;
		}
		public void run() {
			try {
				PackageViewPart pv = (PackageViewPart)getSite().getPage().showView("edu.duke.cs.snarfer.eclipse.PackageView"); //$NON-NLS-1$
				pv.setPackageInfo(info);
			}
			catch (PartInitException e) {
				e.printStackTrace();
			}
		}
	}
	private class RefreshAction extends Action {
		public void run() {
			onRefresh();
		}
	}
	private class UpdateSearchAction extends Action {
		private URL url;
		public UpdateSearchAction() {
			super("Search for Updates"); //$NON-NLS-1$
		}
		public void run() {
			UpdateChecker checker = new UpdateChecker(getSite().getShell(),
				SnarferPlugin.getDefault().getSiteURLs(),
				SnarferPlugin.getDefault().getHistory());
			try {
				new ProgressMonitorDialog(getSite().getShell()).run(true, true, checker);
			} catch (InvocationTargetException e) {
				e.printStackTrace();
			} catch (InterruptedException e) {
				e.printStackTrace();
			}
			if (checker.getUpdates()!=null && checker.getUpdates().size() > 0) {
				UpdateDialog dlg = new UpdateDialog(getSite().getShell(), checker.getUpdates(), checker.getSiteMap());
				dlg.open();
			}
			else {
				MessageDialog.openInformation(getSite().getShell(), Strings.getString("SiteViewPart.search.nonefound.title"), Strings.getString("SiteViewPart.search.nonefound.text")); //$NON-NLS-1$ //$NON-NLS-2$
			}
		}
	}
	private void createToolbar() {
		IToolBarManager mgr = getViewSite().getActionBars().getToolBarManager();
		mgr.add(newSiteAction);
		mgr.add(editSiteAction);
		mgr.add(removeSiteAction);
		mgr.add(openAction);
		mgr.add(installAction);
		mgr.add(refreshAction);
		mgr.add(updateSearchAction);
	}
	private void createContextMenu() {
		// Create menu manager.
		MenuManager menuMgr = new MenuManager();
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener() {
			public void menuAboutToShow(IMenuManager mgr) {
				fillContextMenu(mgr);
			}
		});

		// Create menu.
		Menu menu = menuMgr.createContextMenu(viewer.getControl());
		viewer.getControl().setMenu(menu);

		// Register menu for extension.
		getSite().registerContextMenu(menuMgr, viewer);
	}
	private void fillContextMenu(IMenuManager mgr) {
		mgr.add(newSiteAction);
		mgr.add(refreshAction);
		if (editSiteAction.isEnabled())
			mgr.add(editSiteAction);
		if (removeSiteAction.isEnabled())
			mgr.add(removeSiteAction);
		if (openAction.isEnabled())
			mgr.add(openAction);
		if (installAction.isEnabled())
			mgr.add(installAction);
		mgr.add(new GroupMarker(IWorkbenchActionConstants.MB_ADDITIONS));
	}
	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
	 */
	public void selectionChanged(SelectionChangedEvent event) {
		Object sel = ((IStructuredSelection)event.getSelection()).getFirstElement();		
		if (contentProvider.getParent(sel) == null) {
			URL url = contentProvider.getSiteURL(sel);
			if (url != null) {
				removeSiteAction.setURL(url);
				editSiteAction.setURL(url);
				editSiteAction.setEnabled(true);
				removeSiteAction.setEnabled(true);
			} else {
				editSiteAction.setEnabled(false);
				removeSiteAction.setEnabled(false);
			}
		}
		else {
			removeSiteAction.setEnabled(false);
			editSiteAction.setEnabled(false);
		}		
		if (sel instanceof PackageInfo) {
			openAction.setPackageInfo((PackageInfo)sel);
			openAction.setEnabled(true);
			installAction.setPackageInfo((PackageInfo)sel);
			installAction.setEnabled(true);
		}
		else
		{
			openAction.setEnabled(false);	
			installAction.setEnabled(false);
		}
			
	}
	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e) {
		if (openAction.isEnabled())
			openAction.run();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e) {
	}
}
